
; DC Motor Controller (end of 2016)

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1

; code protection
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3  & _DEBUG_OFF & _WRT_PROTECT_ALL & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _EXTRC_IO  
;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

EEPROM0		equ H'00'	; storage of maximum speed setting
EEPROM1		equ	H'01'	; storage of shutdown mode (shutdown remains till power off or till speed control set at 0)		
	
; bank 0 RAM
ERROR_VAL	equ	H'20'	; error value between required speed and actual
MAX_SET		equ	H'21'	; maximum setting of speed
SPEED_READ	equ	H'22'	; current speed pot reading
SOFT			equ	H'23'	; soft start setting
SPEED			equ	H'24'	; speed (this is the current speed when less than SPEED_READ): SPEED_READ transferred to SPEED when SPEED > SPEED READ 
FEED_BACK	equ	H'25'	; feedback signal
GAIN			equ	H'26'	; feedback gain
CONVERSION	equ	H'27'	; A/D conversion delay counter
STORE1			equ	H'28'	; delay storage	
STORE2			equ	H'29'	; delay storage
FLAG_SS		equ	H'2A'	; flag for soft start
LED_FLAG		equ	H'2B'	; LED indicator flag
H_L_DRIVE		equ	H'2C'	; high or low side drive for back EMF feedback
		
; error values for summation and averaging
E1				equ	H'2E'	; error value 1
E2				equ	H'2F'	; error value 2
E3				equ	H'30'	; error value 3
E4				equ	H'31'	; error value 4
E5				equ	H'32'	; error value 5
E6				equ	H'33'	; error value 6
E7				equ	H'34'	; error value 7
E8				equ	H'35'	; error value 8
ADDN0			equ	H'36'	; error addition ms byte	
ADDN1			equ	H'37'	; error addition ls byte	

TEMP			equ	H'38'	; temporary use
UV_STORE1		equ	H'39'	; undervoltage counter ms byte
UV_STORE2		equ	H'3A'	; undervoltage counter ls byte

; math routines

TEMP1			equ	H'61'
TEMPB1		equ H'62'
TEMPB0      		equ H'63'
TEMPD       		equ H'64'    	; temporary storage
AARGB3		equ	H'65'
AARGB2      		equ H'66'
AARGB1      		equ H'67'
AARGB0      		equ H'68'	; most significant byte of argument A
BARGB1      		equ H'69'
BARGB0      		equ H'6A'	; most significant byte of argument B
LOOPCOUNT   	equ H'6B'   	; loop counter
REMB0			equ	H'6C'	; remainder
REMB1			equ	H'6D'	; remainder	

; All Banks RAM
STATUS_TMP	equ	H'70'	; status storage in interrupt
W_TMP			equ	H'71'	; store w in interrupt
INT_FLG			equ	H'72'	; interrupt flag

	ORG     H'2100'
	DE		D'255'		; set max speed limit at maximum
	DE		D'00'		; shutdown mode (0=shutdown till speed control reduced to 0) (1=shutdown till powered off and on)

; start at memory 0

	org		0			; reset vector
	goto	MAIN		; byte 1

	nop
	nop
	nop	

	org     4				; interrupt vector

INTERRUPT

; usedfor shutdown and to flash SPEED LED
; store w and status registers 
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp
	bcf		STATUS,RP0 
	bcf		STATUS,RP1	; bank 0
 
	btfsc	INTCON,INTF		; if set then shutdown
	goto	DOWN 			; shutdown

FLASH_LED
	bcf		PIR1,TMR1IF	 ; clear interrupt flag for timer1

	btfss	PORTA,6	; check LED on or off
	goto	ON
OFF
	movf	SPEED,w
	movwf	TMR1H		; preset timer with speed value

	bcf		PORTA,6	; LED off
	goto	RECLAIM
ON	
; if zero, don't light LED
	movf	SPEED,w
	btfsc	STATUS,Z
	goto	RECLAIM		
	comf	SPEED,w
	movwf	TMR1H		; preset timer with speed value

	bsf		PORTA,6	; LED on
	goto	RECLAIM	

DOWN ; shutdown
	bcf		INTCON,INTF		; clear flag 
; 0% duty  
	clrf		CCPR1L		; clear PWM
	clrf		SPEED
	bsf		INT_FLG	,0		; interrupt flag
	bcf		PORTA,6		; Speed LED off

; Check shutdown mode
; EEPROM1 shutdown mode (0=shutdown till speed control reduced to 0) (1=shutdown till powered off and on)
	movlw	EEPROM1
	call		EEREAD
	movwf	TEMP

END_POWER ; wait for power off loop
	btfss	TEMP,0			; if set then wait  till power is off
	goto	WAIT_CONV_INTE
	call		FLASH_SD_LED	; flash LED
	goto	END_POWER
	
WAIT_CONV_INTE
	btfsc	ADCON0,2		; conversion complete when cleared 
	goto	WAIT_CONV_INTE	; wait for any A/D conversion to complete

; wait for speed pot to go to zero before exiting interrupt	
	bsf		STATUS,RP0 
	movlw	B'01110000'		; left justified A/D result, Ref- and Ref+ used for speed pot
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11100000' 		; Frc, channel 4 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

; get speed pot value
AD_LOOP
	call		FLASH_SD_LED	; flash LED
	call		ACQUIRE_AD
	movf	ADRESH,w		; check if <2
	sublw	D'02'
	btfss	STATUS,C
	goto	AD_LOOP
 	bcf		PORTB,4	; shutdown LED off

RECLAIM	
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   	W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

FLASH_SD_LED ; subroutine used within interrupt
;  flash shutdown LED
	bsf		PORTB,4	; LED on
	movlw	D'127'
	movwf	TMR1H		; preset timer with speed value
;	wait for timer
WAIT1
	btfss	PIR1,TMR1IF	 ; clear interrupt flag for timer1
	goto	WAIT1
	bcf		PIR1,TMR1IF	 ; clear interrupt flag for timer1
	bcf		PORTB,4	; LED off
	movlw	D'127'
	movwf	TMR1H		; preset timer with speed value
;	wait for timer
WAIT2
	btfss	PIR1,TMR1IF	 ; clear interrupt flag for timer1
	goto	WAIT2
	bcf		PIR1,TMR1IF	 ; clear interrupt flag for timer1

	return

;******************************************************************************************* 
MAIN

	clrf		PORTB			; all low
	clrf		PORTA
	bsf		STATUS,RP0	; select memory bank 1

; set inputs/outputs
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11000111'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'10111111'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000001'	; settings (pullups enabled TMR0/4)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'01111111'		; AN0-7 analog inputs (AN2, AN3 for REF-, REF+)
	movwf	ANSEL
; *
	movlw	B'01110000'		; left justified A/D result, Ref- and Ref+ used for speed pot *
; *
	movlw	B'01000000'		; left justified, references 0V and 5V *
; * use Ref- to REF+ for speed pot, other analog inputs 0 to 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11100000' 		; Frc, channel 1 etc
; bit 5-3
;			000 = Channel 0 (RA0/AN0);
;			001 = Channel 1 (RA1/AN1)
;			010 = Channel 2 (RA2/AN2)
;			011 = Channel 3 (RA3/AN3)
;			100 = Channel 4 (RA4/AN4)
;			101 = Channel 5 (RB6/AN5)
;			110 = Channel 6 (RB7/AN6)

	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		STATUS,RP0	; select memory bank 1

; Timer1
	bsf		PIE1,TMR1IE		; enable timer 1 interrupt
	bcf		STATUS,RP0	; memory bank 0
	movlw	B'10011101'		
	movwf	T1CON			; divider set and  timer 1 enabled

; pwm set
	bsf		STATUS,RP0	; select memory bank 1
	movlw	H'FF'
	movwf	PR2				; PWM period register
	bcf		STATUS,RP0	; memory bank 0
	movlw	D'00'			; 0% duty  
	movwf	CCPR1L		; ms byte of PWM
	movlw	B'00000001'		; prescaler divide by 4
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'		; set PWM mode
	movwf	CCP1CON		; enable PWM operation

; Initial
	clrf		INT_FLG			; interrupt flag
	clrf		FLAG_SS		; flag for soft start
	clrf		LED_FLAG		; LED indicator flag
	clrf		H_L_DRIVE		; high or low side drive for back EMF feedback
	clrf		UV_STORE1		; unvervoltage counter ms byte
	clrf		UV_STORE2		; unvervoltage counter ls byte

; check shutdown. EEPROM1 shutdown mode (0=shutdown till speed control reduced to 0) (1=shutdown till powered off and on)
SHUTDOWN_MODE
; check shutdown alternate switch using limit switch S2  at RB2
	movlw	D'25'
	call		DELAYX		; debounce
	btfss	PORTB,2
	goto	PRESSED
; show MODE
; switch is open
	movlw	EEPROM1
	call		EEREAD
	movwf	TEMP1
	btfss	TEMP1,0	; if zero or 1
	goto	ONE_FLASH
	goto	TWO_FLASH
	
PRESSED
; switch is pressed so toggle shutdown mode
	movlw	EEPROM1
	call		EEREAD
	movwf	TEMP1
	btfss	TEMP1,0	; if zero or 1
	goto	CHNG_1
; change to 0	
	clrw	
	call		EEWRITE_NO_INTERRUPT
; value written
; show one flash of LED4 (shutdown)	
; MODE: shutdown till speed control reduced to 0
ONE_FLASH
	bsf		PORTB,4
	movlw	D'255'
	call		DELAYX
	bcf		PORTB,4
	goto	WAIT_OPEN

CHNG_1 ;  change to 1
	movlw	D'01'
	call		EEWRITE_NO_INTERRUPT
; value written
; show two flashes of LED4 (shutdown)	
; MODE: shutdown till powered off and on
TWO_FLASH
	bsf		PORTB,4
	movlw	D'255'
	call		DELAYX
	bcf		PORTB,4
	movlw	D'255'
	call		DELAYX
	bsf		PORTB,4
	movlw	D'255'
	call		DELAYX
	bcf		PORTB,4

WAIT_OPEN ; wait for shutdown mode switch to opem
	btfss	PORTB,2
	goto	WAIT_OPEN
	movlw	D'255'
	call		DELAYX
	btfss	PORTB,2
	goto	WAIT_OPEN
	
START_DEL	
; startup delay
	movlw	D'1'
	movwf	TEMP1
ST_DEL
	movlw	D'255'
	call		DELAYX
	decfsz	TEMP1,f
	goto	ST_DEL

; read sense input. High (no jumper so input is high is for high side driver)
	btfss	PORTB,1		; sense input
	bsf		H_L_DRIVE,0	; set when low side drive

; read soft start pot at AN0
REDO0
	movlw	B'11000000' 		; Frc, channel 0 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	
	bsf		STATUS,RP0	; select memory bank1
	movlw	B'01000000'		; left justified, references 0V and 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

; Channel 1 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w		 ; soft start
	movwf	SOFT			; soft start setting
; if SOFT=0 set soft start flag
	movf	SOFT,w
	btfsc	STATUS,Z
	bsf		FLAG_SS,0		; set

; allow interrupts
ALL_INTERRUPTS
	bsf		INTCON,INTE	; RB0 interrupt
	bcf		INTCON,INTF		; clear flag
	bcf		PIR1,TMR1IF		; timer1 flag
	bsf		INTCON,PEIE	; peripheral interrupts enabled
	bsf 		INTCON,GIE		; set global interrupt enable 

; retrieve maximum speed setting 
; place EEPROM0 storage into MAX_SET
	movlw	EEPROM0
	call		EEREAD
	movwf	MAX_SET

	clrf		SPEED			; Current speed 	
	clrf		CCPR1L		; pwm off

CYCLE_LOOP

; set A/D to channel 1 to measure battery Voltage
REDO1
	clrf		INT_FLG			; interrupt flag
	movlw	B'11001000' 		; Frc, channel 1 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	
	bsf		STATUS,RP0	; select memory bank1
	movlw	B'01000000'		; left justified, references 0V and 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

; Channel 1 A/D value
	call		ACQUIRE_AD

; check if interrupted
	
	btfsc	INT_FLG	,0		; interrupt flag, if set, redo
	goto	REDO1
	movf	ADRESH,w		 ; battery voltage

; low batt when below 2.5V. Undervoltage stop  if low
	sublw	D'127'
	movlw	D'0'				; ready for a 0
	btfss	STATUS,C

; if carry is clear then voltage ok. If set, undervoltage
	clrf		UV_STORE2		; clear each time over voltage threshold ls byte
	btfss	STATUS,C

; if carry is clear then voltage ok. If set, undervoltage
	clrf		UV_STORE1		; clear each time over voltage threshold ms byte

	movlw	D'01'			; use a 1 if carry is set
	addwf	UV_STORE2,f	; unvervoltage counter (if reaches all 1's then undervoltage)

; if carry is set then add 1 to ms byte
	movlw	D'0'
	btfsc	STATUS,C
	movlw	D'01'
	addwf	UV_STORE1,f	

; check  both UV_STORE1,2  
; ms byte
	movf	UV_STORE1,w
	sublw	D'64'
	btfsc	STATUS,C		; if greater then under voltage
	goto	OVER_V		; not greater
; ls byte
	movf	UV_STORE2,w
	xorlw	D'255'
	btfsc	STATUS,Z		; if 255 then under voltage
	goto	UNDER_VOLTAGE

OVER_V	
; check if maximum speed setting switch is pressed
	btfsc	PORTB,2
	goto	SEQUENCE	

; place SPEED_READ into MAX_SET
	movf	SPEED_READ,w
	movwf	MAX_SET

; max set store in EEPROM
	movlw	EEPROM0
	call 		EEREAD		; sets EEADR
	movf	MAX_SET,w
	call		EEWRITE

SEQUENCE
SPEED_RUN

; read speed pot
; set A/D to channel 4 to measure speed pot
REDO4
	clrf		INT_FLG			; interrupt flag
	movlw	B'11100000' 		; Frc, channel 4 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	
	bsf		STATUS,RP0	; select memory bank1
	movlw	B'01110000'		; left justified A/D result, Ref- and Ref+ used for speed pot only
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

; Channel 4 A/D value
	call		ACQUIRE_AD

; check if interrupted
	
	btfsc	INT_FLG	,0		; interrupt flag, if set, redo
	goto	REDO4
	movf	ADRESH,w		 ; speed pot
	movwf	SPEED_READ	; current speed pot reading

; set FLAG_SS (flag for soft start) if SPEED_READ is 0.
	movf	SPEED_READ,w
	btfsc	STATUS,Z
	bsf		FLAG_SS,0		; end of soft start

RAMP_LOOP
; if current speed pot reading (SPEED_READ) is less than current speed (SPEED) then place SPEED_READ into SPEED
	movf	SPEED_READ,w
	subwf	SPEED,w
	btfss	STATUS,C
	goto	ZRO_CHK
	movf	SPEED_READ,w
	movwf	SPEED
	bsf		FLAG_SS,0		;  end of soft start
	goto	MAX_CHK
ZRO_CHK
	btfsc	STATUS,Z		; if SPEED_READ=SPEED set soft start flag
	bsf		FLAG_SS,0		; end of soft start

MAX_CHK		

; Speed check against Max set
	movf	SPEED,w
	subwf	MAX_SET,w
	movf	MAX_SET,w		; get maximum value ready
	btfss	STATUS,C		; if negative use MAX_SET 	for SPEED
	movwf	SPEED
	btfss	STATUS,C		; if negative,  end of soft start
	bsf		FLAG_SS,0		; end of soft start	
	btfss	STATUS,C		; if negative, light LED
	bsf		PORTB,4		; speed limit LED
	btfsc	STATUS,C		; if positive clear LED
	bcf		PORTB,4		; speed limit LED

; check if soft start	
	btfss	FLAG_SS,0		; if set then bypass soft start and transfer SPEED_READ to SPEED
	goto	INC_RAMP
	movf	SPEED_READ,w
	movwf	SPEED
; Speed check against Max set
	movf	SPEED,w
	subwf	MAX_SET,w
	movf	MAX_SET,w		; get maximum value ready
	btfss	STATUS,C		; if negative use MAX_SET 	for SPEED
	movwf	SPEED
	btfss	STATUS,C		; if negative, light LED
	bsf		PORTB,4		; speed limit LED
	btfsc	STATUS,C		; if positive clear LED
	bcf		PORTB,4		; speed limit LED
	goto	FEEDBACK_RUN

INC_RAMP
; use soft start setting to ramp up speed over time
; increase SPEED up to SPEED_READ
	movlw	D'01'
	addwf	SPEED,f
; add delay based on soft start rate value
	movf	SOFT,w
; check if zero
	btfss	STATUS,Z
	call		DELAYS			; run delay but  bypass soft start if SOFT =  zero
	
; Speed check against Max set
	movf	SPEED,w
	subwf	MAX_SET,w
	movf	MAX_SET,w		; get maximum value ready
	btfss	STATUS,C		; if negative use MAX_SET 	for SPEED
	movwf	SPEED
	btfss	STATUS,C		; if negative, end of soft start
	bsf		FLAG_SS,0		; end of soft start	
	btfss	STATUS,C		; if negative, light LED
	bsf		PORTB,4		; speed limit LED
	btfsc	STATUS,C		; if positive clear LED
	bcf		PORTB,4		; speed limit LED

FEEDBACK_RUN
	btfss	FLAG_SS,0		; check for end of soft start
	goto	SET_PWM 		; set pwm to SPEED

; read feedback and gain

REDO5
	clrf		INT_FLG			; interrupt flag
	movlw	B'11101000' 		; Frc, channel 5 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	
	bsf		STATUS,RP0	; select memory bank1
	movlw	B'01000000'		; left justified, references 0V and 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

; Channel 5 A/D value
	call		ACQUIRE_AD

; check if interrupted
	
	btfsc	INT_FLG	,0		; interrupt flag, if set, redo
	goto	REDO5

; get  feedback value
	bsf		STATUS,RP0	; select memory bank1
	bcf		STATUS,C
	btfsc	ADRESL,7
	bsf		STATUS,C		; set carry when bit 7 is set
	bcf		STATUS,RP0	; select memory bank 0
	rlf		ADRESH,w		; x 2 and gain extra lsb via ADRESL.  Multiply since feedback voltage is attenuated from 0-5V to 0-2.5V

	movwf	FEED_BACK

; High or low side switching (HSS or LSS)
; check HSS, LSS flag
	btfss	H_L_DRIVE,0	; bit 0 is  set when low side drive
	goto	AN6				; HSS

LSS
; when low side drive, process feedback signal. LSS feedback works opposite way to HSS version. ie 2.5V-0V instead of 0-2.5V

	movf	FEED_BACK,w
	sublw	D'127'			; take from 2.5V
	movwf	FEED_BACK
	btfss	STATUS,C
	clrf		FEED_BACK
	bcf		STATUS,C
	rlf		FEED_BACK,f

AN6
; read AN6 feedback gain
REDO6
	clrf		INT_FLG			; interrupt flag
	movlw	B'11110000' 		; Frc, channel 6 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	
	bsf		STATUS,RP0	; select memory bank1
	movlw	B'01000000'		; left justified, references 0V and 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
;
; Channel 1 A/D value
	call		ACQUIRE_AD

; check if interrupted
	
	btfsc	INT_FLG	,0		; interrupt flag, if set, redo
	goto	REDO6
	movf	ADRESH,w		 ;
	movwf	GAIN			;  setting for gain

; if  GAIN is less than 2 ignore feedback control

	sublw	D'02'
	btfsc	STATUS,C
	goto	SET_PWM
	movlw	D'02'
	subwf	GAIN,f		; gain - 2
	incf		GAIN,f		; minimum of 1
; compare SPEED to FEEDBACK. Use gain (AN6) setting to adjust feedback rate
	bcf		STATUS,C
	rrf		SPEED,w

	subwf	FEED_BACK,w	; speed measured
	btfsc	STATUS,Z		; if the same, exit any change
	goto	SET_PWM		;CYCLE_LOOP

	btfss	STATUS,C		; if speed  is larger than FEED_BACK increase PWM 
	goto	INC_PWM

SET_PWM ; set pwm to SPEED
	movf	SPEED,w
	movwf	CCPR1L		; ms byte of PWM
	clrf		E8
	clrf		E7
	clrf		E6
	clrf		E5
	clrf		E4
	clrf		E3
	clrf		E2
	clrf		E1
	goto	CYCLE_LOOP

INC_PWM
	bcf		STATUS,C
	rrf		SPEED,w

	movwf	TEMP1

	movf	FEED_BACK,w
	subwf	TEMP1,w
	btfsc	STATUS,Z		; if zero use 1	
	movlw	D'1'
	movwf	ERROR_VAL
AVERAGE
; average out error readings
; First move stored error values along 

	movf	E7,w
	movwf	E8
	movf	E6,w
	movwf	E7
	movf	E5,w
	movwf	E6
	movf	E4,w
	movwf	E5
	movf	E3,w
	movwf	E4
	movf	E2,w
	movwf	E3
	movf	E1,w
	movwf	E2
	movf	ERROR_VAL,w
	movwf	E1

; add then up into ADDN0,ADDN1
	clrf		ADDN0
	movf	E1,w
	movwf	ADDN1
	movf	E2,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E3,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E4,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E5,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E6,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E7,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	E8,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f

; divide by 8
;/2
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,f
;/4
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,f
;/8
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,w
	movwf	ERROR_VAL
	movf	ERROR_VAL,w
	btfsc	STATUS,Z		; if zero use 1	
	movlw	D'1'
	movwf	ERROR_VAL	;  average error
	
;   alter PWM according to feedback  and gain setting 
ALTER
; multiply GAIN by ERROR_VAL (a 16-bit result in AARGB2 and AARGB3)
	clrf		AARGB0		; clear
	movf	GAIN,w
	movwf	AARGB1
;  if  zero, disable feedback
	btfsc	STATUS,Z
	goto	SET_PWM 	; set pwm to SPEED

	clrf		BARGB0

	movf	ERROR_VAL,w
	movwf	BARGB1
	call		FXM1616U	; multiply 

;  result in AARGB2,3 (AARGB0,1 should be 0)	

	movf	AARGB1,w	; if AARGB1 is not zero
	btfss	STATUS,Z
	goto	MAX_RATE

; move the values along for division
	movf	AARGB2,w
	movwf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1

; if all 0 use SPEED
	movf	AARGB0,w
	btfss	STATUS,Z
	goto	DIV1
	movf	AARGB1,w
	btfss	STATUS,Z
	goto	DIV1

	movf	SPEED,w
; if speed is zero clear PWM
	btfsc	STATUS,Z
	goto	CLEAR_PWM

DIV1
; divide value,   
	movlw	D'30'		 ; sets overall PWM increase with load. Smaller value = greater effect (16 bit  BARGB0, BARGB1)
	movwf	BARGB0
	call		DIV16_8		; divide by BARGB0

;  result in AARGB0,AARGB1
; check AARGB0 
	movf	AARGB0,w	; ms
	btfss	STATUS,Z	; if not zero then use max rate for CCPR1L
	goto	MAX_RATE		
	
	movf	AARGB1,w
	addwf	SPEED,w	; for feedback add calculation to speed
	btfsc	STATUS,C
	goto	MAX_RATE	; if overrange use maximum

	movwf	CCPR1L	; ms byte of PWM
	goto	CYCLE_LOOP

MAX_RATE
	movlw	D'255'
; if speed is used then this is a current limit
;	movf	SPEED,w	; if overrange use speed
	movwf	CCPR1L	; ms byte of PWM
	goto	CYCLE_LOOP
CLEAR_PWM
	clrf		CCPR1L
	goto	CYCLE_LOOP

; >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

UNDER_VOLTAGE
; switch on UV LED (LED3)	
	bsf		PORTB,5
; cease output
	clrf		CCPR1L		; ms byte of PWM
	clrf		SPEED
; stop here

STOP
	goto	STOP			; only exited when powered off and on again

; xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

; Subroutines

; subroutine to wait for conversion

ACQUIRE_AD
; wait for >20us
	movlw	D'25'
	movwf	CONVERSION
LOOP_CONV
	decfsz	CONVERSION,f	; decrease 
	goto	LOOP_CONV	
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2		; conversion complete when cleared 
	goto	WAIT_CONV
	return

DELAYms
	movlw	D'23'		; delay value
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'70'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8		; decrease till STORE1 is zero
	return

DELAYS
	movwf	STORE1
DELAYS1
	movlw	D'30'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP19
	decfsz	STORE2,f
	goto	LOOP19

	decfsz	STORE1,f
	goto	DELAYS1	

; subroutine to read EEPROM memory 

EEREAD
	
	bcf		STATUS,RP0	; select bank 0
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bsf 		STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD		; read EEPROM
	bcf 		STATUS,RP0	; select memory bank 2
	movf	EEDATA,w		; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return


EEWRITE
; subroutine to write to EEPROM interrupt enabled	

	bcf 		STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 		STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE		; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'			; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'			; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 		INTCON,GIE		; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE			; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 		STATUS,RP0	; select memory bank 0
	return				 	; value written 

EEWRITE_NO_INTERRUPT
; subroutine to write to EEPROM interrupt enabled	

	bcf 		STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 		STATUS,RP0	; select memory bank 3
WRX	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WRX			; not written yet
	bcf		INTCON,GIE		; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'			; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'			; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit

	goto	WRITE

	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8 
		CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
				return

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LA 		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LC 		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L			RETURN
	

	end
